// File.cpp: implementation of the CFile class.
//
//////////////////////////////////////////////////////////////////////

#include "pch.h"
#include "File.h"

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

CFile::CFile() :
	m_hFile(INVALID_HANDLE_VALUE)
{

}

CFile::~CFile()
{
	CloseHandle( m_hFile );
}

//////////////////////////////////////////////////////////////////////
// Methods
//////////////////////////////////////////////////////////////////////

bool CFile::Open(const tstring &sFileName, 
			DWORD dwFileAccess,
			DWORD dwCreationDisposition,
			DWORD dwFlagsAndAttributes )
{
	if (m_hFile != INVALID_HANDLE_VALUE)
		::CloseHandle( m_hFile );
	m_hFile = ::CreateFile( sFileName.c_str(), dwFileAccess, 0, NULL, dwCreationDisposition, dwFlagsAndAttributes, NULL );
	return (m_hFile != INVALID_HANDLE_VALUE);
}

bool CFile::Close() {
	if (CloseHandle(m_hFile)) {
		m_hFile = INVALID_HANDLE_VALUE;
		return (true);
	}
	return (false);
}

bool CFile::Write(LPCVOID lpData, DWORD dwLen) {
	if (m_hFile == INVALID_HANDLE_VALUE)
		return (false);
	DWORD	dwBytesWritten;
	if (!WriteFile( m_hFile, lpData, dwLen, &dwBytesWritten, NULL) || dwBytesWritten != dwLen)
		return (false);
	return (true);
}

bool CFile::WriteStringAsOem(LPCTSTR szData) {
	int	len = lstrlen(szData);
	if (len > MAX_PATH-1) {
		char*	szOemData;
		szOemData = new char[len+1];
		CharToOem( szData, szOemData );
		return (Write( szOemData, len ));
		delete [] szOemData;
	}
	char	szOemData[MAX_PATH];
	CharToOem( szData, szOemData );
	return (Write( szOemData, len ));
}

bool CFile::WriteOemString(const char* szData) {
	return (Write( szData, strlen(szData) ));
}

bool CFile::SeekToEnd() {
	return (SetFilePointer( m_hFile, 0, NULL, FILE_END ) != 0xFFFFFFFF);
}


/**
 * COemLineReader class implementation
 */


COemLineReader::COemLineReader(LPCTSTR fileName) {
	fileHandle = ::CreateFile( fileName, GENERIC_READ, 0, NULL, OPEN_EXISTING, 
							   FILE_ATTRIBUTE_NORMAL | FILE_FLAG_SEQUENTIAL_SCAN, NULL );
	if (isOpened()) {
		bufferSize = 2048;
		buffer     = new char[bufferSize];
		line       = new char[bufferSize+1];

		// no data yet

		startPos = 0;
		dataSize = 0;
		eof      = false;
	}
}


COemLineReader::~COemLineReader() {	
	if (isOpened()) {
		delete [] line;
		delete [] buffer;
		::CloseHandle( fileHandle );
	}
}


bool COemLineReader::next() {
	if (!isOpened()) {
		return false;
	}

	if (dataSize == 0) {
		// no data in buffer, let's read if not eof
		if (eof) {
			return false;
		}

		startPos = 0;

		if (::ReadFile( fileHandle, buffer, bufferSize, &dataSize, NULL ) == 0) {
			// error occured
			// TODO: log error
			eof = true;
			return false;
		}

		if (dataSize == 0) {
			// no more data in file
			eof = true;
			return false;
		}
	}

	// at this moment we have some data in buffer
	// we use line separator CRLF

	// let's try to find it in buffer from startPos during dataSize

	int	eolPos;
	int	eolSearchOffset = 0;
	do {
		eolPos = getEolPos(eolSearchOffset);
		if (eolPos < 0) {
			// not found
			// let's try to read more data
			eolSearchOffset = dataSize;
			if (eof || !readMore()) {
				// there is last string in buffer
				eolPos = startPos + dataSize;
			}
		}
	} while (eolPos < 0);

	// our line is from startPos to eolPos-1
	rememberLine(eolPos);

	// correct tail size and position
	int	eolSize = 0;
	if (eolPos < (startPos + dataSize)) {
		if (buffer[eolPos] == 0x0D)
			eolSize = 2;	// CRLF
		else
			eolSize = 1;	// LF
	}
	eolPos += eolSize;
	dataSize -= (eolPos-startPos);
	startPos = eolPos;

	return true;
}


LPCSTR COemLineReader::getLine() {
	if (!isOpened()) {
		return NULL;
	}	
	return line;
}


int COemLineReader::getEolPos(int eolSearchOffset) {
	int	pos = startPos + eolSearchOffset;
	int	endPos = startPos + dataSize;

	while ((pos < endPos) && (buffer[pos] != 0x0A)) {
		pos++;
	}

	if (pos == endPos) {
		// end-of-line not found
		return -1;
	}

	if ((pos > startPos) && (buffer[pos-1] == 0x0D)) {
		pos--;
	}

	return pos;
}


bool COemLineReader::readMore() {

	// we will enlarge buffer if dataSize is greater then bufferSize/2

	if (dataSize > (bufferSize >> 1)) {
		bufferSize <<= 1;
		LPSTR	newBuffer = new char[bufferSize];
		::CopyMemory( newBuffer, buffer+startPos, dataSize );
		startPos = 0;
		delete [] buffer;
		buffer = newBuffer;
		delete [] line;
		line = new char[bufferSize+1];
	}
	else {
		// shift data to the start of the buffer
		if (startPos > 0) {
			::MoveMemory( buffer, buffer + startPos, dataSize );
			startPos = 0;
		}
	}

	DWORD	dataRead;
	if (::ReadFile( fileHandle, buffer + startPos + dataSize, bufferSize >> 1, &dataRead, NULL ) == 0) {
		// error occured
		// TODO: log error
		eof = true;
		return false;
	}
	if (dataRead == 0) {
		// no more data
		eof = true;
		return false;
	}
	dataSize += dataRead;

	return true;
}


void COemLineReader::rememberLine(int eolPos) {
	if (eolPos > startPos) {
		::CopyMemory( line, buffer + startPos, eolPos - startPos );
	}
	line[eolPos - startPos] = '\0';
}
